// contexts/MapStateContext.tsx
"use client";

import { createContext, useContext, useState, useCallback } from "react";
import { useSearchParams, usePathname, useRouter } from "next/navigation";

export interface MapState {
  center: [number, number];
  zoom: number;
  openPopupId: string | null;
}

interface MapStateContextType {
  mapState: MapState;
  setMapState: (state: React.SetStateAction<MapState>) => void;
  syncStateToURL: () => void;
}

const MapStateContext = createContext<MapStateContextType | undefined>(undefined);

export const MapStateProvider = ({ children }: { children: React.ReactNode }) => {
  const searchParams = useSearchParams();
  const pathname = usePathname();
  const router = useRouter();
  
  const [mapState, setMapState] = useState<MapState>(() => {
    // Initialize from URL if available
    const lat = searchParams.get('lat');
    const lng = searchParams.get('lng');
    const zoom = searchParams.get('zoom');
    const popup = searchParams.get('popup');
    
    return {
      center: lat && lng ? 
        [parseFloat(lat), parseFloat(lng)] : 
        [51.505, -0.09],
      zoom: zoom ? parseInt(zoom) : 14,
      openPopupId: popup || null,
    };
  });

  const syncStateToURL = useCallback(() => {
    const params = new URLSearchParams(searchParams.toString());
    params.set('lat', mapState.center[0].toString());
    params.set('lng', mapState.center[1].toString());
    params.set('zoom', mapState.zoom.toString());
    
    if (mapState.openPopupId) {
      params.set('popup', mapState.openPopupId);
    } else {
      params.delete('popup');
    }
    
    router.replace(`${pathname}?${params.toString()}`, { scroll: false });
  }, [mapState, searchParams, pathname, router]);

  return (
    <MapStateContext.Provider value={{ mapState, setMapState, syncStateToURL }}>
      {children}
    </MapStateContext.Provider>
  );
};

export const useMapState = () => {
  const context = useContext(MapStateContext);
  if (!context) {
    throw new Error("useMapState must be used within a MapStateProvider");
  }
  return context;
};