import {
  format,
  parseISO,
  isSameDay,
  isToday,
  isTomorrow,
  isWithinInterval,
  startOfMonth,
  endOfMonth,
  addMonths,
  
} from "date-fns";
import { fr,enUS } from "date-fns/locale";

const normalizeDateString = (dateString: string) => {
  const parts = dateString.split("-");
  if (parts.length === 3) {
    const [year, month, day] = parts;
    const normalizedMonth = month.padStart(2, "0");
    const normalizedDay = day.padStart(2, "0");
    return `${year}-${normalizedMonth}-${normalizedDay}`;
  }
  return dateString; // Return original if it doesn't match the expected format
};

const formatdetailsDate = (dateString: string) => {
  const normalizedDate = normalizeDateString(dateString);

  const date = parseISO(normalizedDate);
  return format(date, "d MMMM", { locale: fr });
};

const formatTime = (timeString: string) => {
  const [hours, minutes] = timeString.split(":");
  return minutes === "00"
    ? `${parseInt(hours, 10)}h`
    : `${parseInt(hours, 10)}h${minutes}`;
};
export const getShortMonthName = (date: Date, locale: Locale): string => {
  const fullMonth = format(date, 'MMMM');
  return fullMonth.substring(0, 3);
};
const formatEventDateTime = (
  startDate: string,
  startTime: string = "00:00",
  endDate: string,
  endTime: string = "00:00"
): string => {
  if (!startDate || !endDate) return "Schedule unavailable";

  try {
    // Parse dates safely
    const parseSafe = (dateStr: string, timeStr: string): Date => {
      const datetimeStr = `${dateStr}T${
        timeStr.includes(":") ? timeStr : "00:00"
      }`;
      const date = parseISO(datetimeStr);
      if (isNaN(date.getTime())) throw new Error("Invalid date");
      return date;
    };
    let start = parseSafe(startDate, startTime);
    let end = parseSafe(endDate, endTime);
    let swapped = false;
    if (start > end) {
      [start, end] = [end, start];
      [startDate, endDate] = [endDate, startDate];
      [startTime, endTime] = [endTime, startTime];
      swapped = true;
    }
    const now = new Date();
    let label = "";
    if (isWithinInterval(now, { start, end })) label = "En cours • ";
    else if (isToday(start)) label = "Aujourd'hui • ";
    else if (isTomorrow(start)) label = "Demain • ";
    if (!isSameDay(start, end)) {
      return label
        ? `${label}${formatdetailsDate(startDate)}, ${formatTime(
            startTime
          )} - ${formatdetailsDate(endDate)}, ${formatTime(endTime)}`
        : `${formatdetailsDate(startDate)}, ${formatTime(
            startTime
          )} - ${formatdetailsDate(endDate)}, ${formatTime(endTime)}`;
    }

    if (label) {
      return `${label}${formatTime(startTime)} - ${formatTime(endTime)}`;
    }

    return `${formatdetailsDate(startDate)}, ${formatTime(
      startTime
    )} - ${formatTime(endTime)}`;
  } catch (error) {
    return "Schedule information unavailable";
  }
};
// Helper function to normalize French characters (remove accents)
export const normalizeFrenchText = (text: string): string => {
  if (!text) return '';
  
  return text
    .normalize('NFD')
    .replace(/[\u0300-\u036f]/g, '')
    .toLowerCase();
};
const getMonthNames = (language: string) => {
  const englishMonths = [
    "jan",
    "feb",
    "mar",
    "apr",
    "may",
    "jun",
    "jul",
    "aug",
    "sep",
    "oct",
    "nov",
    "dec",
  ];

  const frenchMonths = [
    "jan",
    "fév",
    "mars",
    "avr",
    "mai",
    "juin",
    "juill",
    "août",
    "sept",
    "oct",
    "nov",
    "déc",
  ];

  return language === "fr" ? frenchMonths : englishMonths;
};
// utils/dateHelpers.ts

// Define the type for date filter keys
export type DateFilterKey = 
  | 'aujourd\'hui' 
  | 'demain' 
  | 'ce weekend' 
  | '+7 jours' 
  | '+15 jours' 
  | 'ce mois'  // Changed from 'en juin' to dynamic 'ce mois'
  | 'mois prochain'  // Changed from 'en juille' to dynamic 'mois prochain'
  | 'tous';


// Define the type for date filter values
export interface DateFilterValue {
  label: string;
  startDate: string;
  endDate: string;
  display: string;
}

// Define the type for the date filters object
export type DateFilters = {
  [key in DateFilterKey]: DateFilterValue;
};

// French month names
const frenchMonths = [
  "jan", "fév", "mars", "avr", "mai", "juin", 
  "juill", "août", "sept", "oct", "nov", "déc"
];

// Get France timezone date
export const getFranceDate = (date = new Date()): Date => {
  const offset = date.getTimezoneOffset() / 60;
  const franceOffset = date.getTimezoneOffset() === 60 ? 0 : 1;
  return new Date(date.getTime() + (offset + franceOffset) * 60 * 60 * 1000);
};

// Format date for API (YYYY-MM-DD)
export const formatDateForAPI = (date: Date): string => {
  return date.toISOString().split('T')[0];
};
const getMonthNameFromArray = (date: Date, language: string): string => {
  const monthIndex = date.getMonth(); // 0-11
  const monthNames = getMonthNames(language);
  return monthNames[monthIndex];
};

// Date range calculations with proper typing
export const getDateRanges = (language: string): DateFilters => {
  const isFrench = language === "fr"; 
  const today = getFranceDate();
  today.setHours(0, 0, 0, 0);
  
  const tomorrow = getFranceDate(today);
  tomorrow.setDate(today.getDate() + 1);

  // This weekend (Saturday to Sunday)
  const thisWeekendStart = getFranceDate(today);
  thisWeekendStart.setDate(today.getDate() + (6 - today.getDay()));
  const thisWeekendEnd = getFranceDate(thisWeekendStart);
  thisWeekendEnd.setDate(thisWeekendStart.getDate() + 1);

  // +7 days from today
  const jPlus7Start = getFranceDate(today);
  const jPlus7End = getFranceDate(today);
  jPlus7End.setDate(today.getDate() + 7);

  // +15 days from today
  const jPlus15Start = getFranceDate(today);
  const jPlus15End = getFranceDate(today);
  jPlus15End.setDate(today.getDate() + 15);
  // Current month (dynamic)
  const thisMonthStart = startOfMonth(today);
  const thisMonthEnd = endOfMonth(today);
  const thisMonthName = getMonthNameFromArray(today, language);

  // Next month (dynamic)
  const nextMonthStart = startOfMonth(addMonths(today, 1));
  const nextMonthEnd = endOfMonth(addMonths(today, 1));
  const nextMonthName = getMonthNameFromArray(addMonths(today, 1), language);

  return {
    'tous': {
      label: isFrench ? 'tous' : 'all',
      startDate: '',
      endDate: '',
      display: isFrench ? 'tous' : 'all'
    },
    'aujourd\'hui': { 
      label: isFrench ? 'aujourd\'hui' : 'today',
      startDate: formatDateForAPI(today), 
      endDate: formatDateForAPI(today),
      display: isFrench ? 'aujourd\'hui' : 'today'
    },
    'demain': { 
      label: isFrench ? 'demain' : 'tomorrow',
      startDate: formatDateForAPI(tomorrow), 
      endDate: formatDateForAPI(tomorrow),
      display: isFrench ? 'demain' : 'tomorrow'
    },
    'ce weekend': { 
      label: isFrench ? 'ce weekend' : 'this weekend',
      startDate: formatDateForAPI(thisWeekendStart), 
      endDate: formatDateForAPI(thisWeekendEnd),
      display: isFrench ? 'ce weekend' : 'this weekend'
    },
    '+7 jours': { 
      label: isFrench ? '+7 jours' : '+7 days',
      startDate: formatDateForAPI(jPlus7Start), 
      endDate: formatDateForAPI(jPlus7End),
      display: isFrench ? '+7 jours' : '+7 days'
    },
    '+15 jours': { 
      label: isFrench ? '+15 jours' : '+15 days',
      startDate: formatDateForAPI(jPlus15Start), 
      endDate: formatDateForAPI(jPlus15End),
      display: isFrench ? '+15 jours' : '+15 days'
    },
    'ce mois': { 
      label: isFrench ? 'ce mois' : 'this month',
      startDate: formatDateForAPI(thisMonthStart),
      endDate: formatDateForAPI(thisMonthEnd),
    display: isFrench ? `en ${thisMonthName}` : `in ${thisMonthName}`
    },
    'mois prochain': { 
      label: isFrench ? 'mois prochain' : 'next month',
      startDate: formatDateForAPI(nextMonthStart), 
      endDate: formatDateForAPI(nextMonthEnd),
       display: isFrench ? `en ${nextMonthName}` : `in ${nextMonthName}`
    }
};
}
// Filter events by date range
export const filterEventsByDate = (events: any[], startDate: string, endDate: string) => {
  if (!startDate || !endDate) return events;
  
  const start = new Date(startDate);
  const end = new Date(endDate);
  
  return events.filter(event => {
    if (!event.start_date) return false;
    
    const eventStartDate = new Date(event.start_date);
    const eventEndDate = event.end_date ? new Date(event.end_date) : eventStartDate;
    
    return (eventStartDate <= end && eventEndDate >= start);
  });
};
const formatEventDateTimeSelected = (event:any) => {
  if (!event.start_date) return "";
  
  const formatTime = (timeString:any) => {
    if (!timeString) return "";
    // Remove seconds and format as HHhMM
    return timeString.slice(0, 5).replace(':', 'h');
  };

  const formatDate = (dateString:any) => {
    if (!dateString) return "";
    
    const date = new Date(dateString);
    const days = ['dim', 'lun', 'mar', 'mer', 'jeu', 'ven', 'sam'];
    const months = ['jan', 'fév', 'mar', 'avr', 'mai', 'jun', 'jui', 'aoû', 'sep', 'oct', 'nov', 'déc'];
    
    const dayName = days[date.getDay()];
    const day = date.getDate();
    const month = months[date.getMonth()];
    
    return `${dayName} ${day} ${month}`;
  };

  const startDateFormatted = formatDate(event.start_date);
  const startTimeFormatted = formatTime(event.time);
  
  return `${startDateFormatted} | ${startTimeFormatted}`;
};

const formatTodayTime = (event :any) => {
  if (!event.time) return "today";
  
  const timeFormatted = event.time.slice(0, 5).replace(':', 'h');
  return `today | ${timeFormatted}`;
};

export { formatEventDateTime, formatdetailsDate, formatTime, getMonthNames,formatEventDateTimeSelected,formatTodayTime };
