// utils/localStorageUtils.ts
import L from "leaflet";
export const STORAGE_KEYS = {
  VIEWED_EVENTS: "viewedEvents",
  VIEWED_SELECTED_EVENTS: "viewedSelectedEvents",
};
export const ZOOM_STORAGE_KEY = "leaflet_map_zoom";
export const CENTER_STORAGE_KEY = "leaflet_map_center";
export interface ViewedEvent {
  eventId: string | number;
  timestamp: number;
  eventTitle: string;
  city: string;
}
interface StoredZoomData {
  zoom: number;
  timestamp: number;
}
interface StoredCenterData {
  lat: number;
  lng: number;
  timestamp: number;
}
export const getViewedEvents = (): ViewedEvent[] => {
  if (typeof window === "undefined") return [];

  try {
    const stored = localStorage.getItem(STORAGE_KEYS.VIEWED_EVENTS);
    return stored ? JSON.parse(stored) : [];
  } catch (error) {
    console.error("Error reading viewed events from localStorage:", error);
    return [];
  }
};
export const getViewedSelectedEvents = (): ViewedEvent[] => {
  if (typeof window === "undefined") return [];

  try {
    const stored = localStorage.getItem(STORAGE_KEYS.VIEWED_SELECTED_EVENTS);
    return stored ? JSON.parse(stored) : [];
  } catch (error) {
    console.error("Error reading viewed events from localStorage:", error);
    return [];
  }
};

export const addViewedEvent = (event: Omit<ViewedEvent, "timestamp">): void => {
  if (typeof window === "undefined") return;

  try {
    const viewedEvents = getViewedEvents();
    const now = Date.now();
    const twentyFourHours = 24 * 60 * 60 * 1000;

    // Filter out events older than 24 hours
    const recentEvents = viewedEvents.filter(
      (item) => now - item.timestamp < twentyFourHours
    );

    // Check if event already exists
    const existingIndex = recentEvents.findIndex(
      (item) => item.eventId === event.eventId
    );

    if (existingIndex !== -1) {
      // Update timestamp if exists
      recentEvents[existingIndex].timestamp = now;
    } else {
      // Add new event with timestamp
      recentEvents.push({
        ...event,
        timestamp: now,
      });
    }

    localStorage.setItem(
      STORAGE_KEYS.VIEWED_EVENTS,
      JSON.stringify(recentEvents)
    );
  } catch (error) {
    console.error("Error saving viewed event to localStorage:", error);
  }
};
export const addSelectedViewedEvent = (
  event: Omit<ViewedEvent, "timestamp">
): void => {
  if (typeof window === "undefined") return;

  try {
    const viewedEvents = getViewedSelectedEvents();
    const now = Date.now();
    const twentyFourHours = 24 * 60 * 60 * 1000;

    // Filter out events older than 24 hours
    const recentEvents = viewedEvents.filter(
      (item) => now - item.timestamp < twentyFourHours
    );

    // Check if event already exists
    const existingIndex = recentEvents.findIndex(
      (item) => item.eventId === event.eventId
    );

    if (existingIndex !== -1) {
      // Update timestamp if exists
      recentEvents[existingIndex].timestamp = now;
    } else {
      // Add new event with timestamp
      recentEvents.push({
        ...event,
        timestamp: now,
      });
    }

    localStorage.setItem(
      STORAGE_KEYS.VIEWED_SELECTED_EVENTS,
      JSON.stringify(recentEvents)
    );
  } catch (error) {
    console.error("Error saving viewed event to localStorage:", error);
  }
};

export const hasViewedEvent = (eventId: string | number): boolean => {
  if (typeof window === "undefined") return false;

  try {
    const viewedEvents = getViewedEvents();
    const now = Date.now();
    const twentyFourHours = 24 * 60 * 60 * 1000;

    return viewedEvents.some(
      (item) =>
        item.eventId === eventId && now - item.timestamp < twentyFourHours
    );
  } catch (error) {
    console.error("Error checking viewed event:", error);
    return false;
  }
};
export const hasViewedSelectedEvent = (eventId: string | number): boolean => {
  if (typeof window === "undefined") return false;

  try {
    const viewedEvents = getViewedSelectedEvents();
    const now = Date.now();
    const twentyFourHours = 24 * 60 * 60 * 1000;

    return viewedEvents.some(
      (item) =>
        item.eventId === eventId && now - item.timestamp < twentyFourHours
    );
  } catch (error) {
    console.error("Error checking viewed event:", error);
    return false;
  }
};
export const storeZoomLevel = (zoom: number): void => {
  const data: StoredZoomData = {
    zoom,
    timestamp: Date.now(),
  };
  localStorage.setItem(ZOOM_STORAGE_KEY, JSON.stringify(data));
};

export const getStoredZoomLevel = (): number | null => {
  try {
    const stored = localStorage.getItem(ZOOM_STORAGE_KEY);
    if (!stored) return null;

    const data: StoredZoomData = JSON.parse(stored);
    const now = Date.now();
    const twentyFourHours = 24 * 60 * 60 * 1000;

    // Check if data is older than 24 hours
    if (now - data.timestamp > twentyFourHours) {
      localStorage.removeItem(ZOOM_STORAGE_KEY);
      return null;
    }

    return data.zoom;
  } catch (error) {
    console.error("Error retrieving stored zoom level:", error);
    return null;
  }
};

export const storeCenter = (center: L.LatLng): void => {
  const data: StoredCenterData = {
    lat: center.lat,
    lng: center.lng,
    timestamp: Date.now(),
  };
  localStorage.setItem(CENTER_STORAGE_KEY, JSON.stringify(data));
};

export const getStoredCenter = (): L.LatLng | null => {
  try {
    const stored = localStorage.getItem(CENTER_STORAGE_KEY);
    if (!stored) return null;

    const data: StoredCenterData = JSON.parse(stored);
    const now = Date.now();
    const twentyFourHours = 24 * 60 * 60 * 1000;

    if (now - data.timestamp > twentyFourHours) {
      localStorage.removeItem(CENTER_STORAGE_KEY);
      return null;
    }

    return L.latLng(data.lat, data.lng);
  } catch (error) {
    console.error("Error retrieving stored center:", error);
    return null;
  }
};

export const cleanupExpiredZoomData = (): void => {
  getStoredZoomLevel();
  getStoredCenter();
};
// Add this to your localstorageutils.ts
export const getLatestViewedEvent = (): {
  eventId: string;
  eventTitle: string;
  city: string;
  timestamp: string;
} | null => {
  if (typeof window === "undefined") return null;

  try {
    const viewedEvents = JSON.parse(
      localStorage.getItem("viewedEvents") || "[]"
    );
    if (viewedEvents.length > 0) {
      // Return the most recent event (last in array)
      return viewedEvents[viewedEvents.length - 1];
    }
    return null;
  } catch (error) {
    console.error("Error getting latest viewed event:", error);
    return null;
  }
};
