import { format, parseISO } from "date-fns";
import { fr, enUS } from "date-fns/locale";

export function formatDateToFrenchLocale(dateString, lang) {
  let formattedDate = "";
  if (dateString) {
    // Parse the input date string to a JavaScript Date object
    const date = parseISO(dateString);

    // Format the date according to the French locale
    if (lang == "fr") {
      formattedDate = format(date, "d MMMM yyyy");
    } else if (lang == "en") {
      formattedDate = format(date, "d MMMM yyyy");
    }
    return formattedDate;
  }
}

export function shortenString(str, maxLength) {
  if (str.length > maxLength) {
    //return str.slice(0, maxLength) + '...';
    return str.slice(0, maxLength);
  } else {
    return str;
  }
}

// utils/numberUtils.js or utils/numberUtils.ts (if using TypeScript)
export function beautifyNumber(number) {
  const abbreviations = {
    12: "T",
    9: "B",
    6: "M",
    3: "K",
  };

  let formattedNumber;
  for (const exponent in abbreviations) {
    if (number >= Math.pow(10, exponent)) {
      const dividedNumber = (number / Math.pow(10, exponent)).toFixed(1);
      formattedNumber = `${dividedNumber.replace(".", ",")}${
        abbreviations[exponent]
      }`;
      break;
    }
  }

  return formattedNumber || number.toString().replace(".", ",");
}

const deg2rad = (deg) => {
  return deg * (Math.PI / 180);
};

const rad2deg = (rad) => {
  return rad * (180 / Math.PI);
};
export function calculateDistanceFromCoordinates(lat1, lon1, lat2, lon2, unit) {
  if (lat1 === lat2 && lon1 === lon2) {
    return 0;
  } else {
    if (lat2 !== "null" && lon2 !== "null") {
      const theta = lon1 - lon2;
      const dist =
        Math.sin(deg2rad(lat1)) * Math.sin(deg2rad(lat2)) +
        Math.cos(deg2rad(lat1)) *
          Math.cos(deg2rad(lat2)) *
          Math.cos(deg2rad(theta));
      let miles = rad2deg(Math.acos(dist)) * 60 * 1.1515;
      unit = unit.toUpperCase();

      if (unit === "K") {
        const rounderNumber = miles * 1.609344;
        return Math.round(rounderNumber.toFixed(2))
          .toString()
          .replace(".", ",");
      } else if (unit === "N") {
        const rounderNumber = miles * 0.8684;
        return Math.round(rounderNumber.toFixed(2))
          .toString()
          .replace(".", ",");
      } else {
        const rounderNumber = miles;
        return Math.round(rounderNumber.toFixed(2))
          .toString()
          .replace(".", ",");
      }
    } else {
      return "--";
    }
  }
}

export function getSlug(text) {
  text = text.replace(/[^\p{L}\d]+/gu, "-");
  text = text.trim("-");
  text = text.toLowerCase();
  // text = text.replace(/[^\-\w]+/g, ''); // This line is commented because it is not present in the PHP code

  if (text === "") {
    return "n-a";
  }

  return text;
}

export const convertTo24HourFormat = (timeString) => {
  if (timeString) {
    const convertedTime = timeString.replace(":", "h");
    return convertedTime;
  } else {
    return "";
  }
 
};
export function extractNumberBeforeColon(inputString) {
  if (inputString) {
    // Use a regular expression to match and extract the number before the colon
    const match = inputString.match(/(\d+):/);

    if (match) {
      // The number before the colon is in the first capturing group (match[1])
      return match[1] + "h";
    } else {
      // If no match is found, return a default value or handle the error as needed
      return "";
    }
  } else {
    return "";
  }
}
export const addAsteriskBeforeAtEmail = (email) => {
  let modifiedEmail = "";
  if (email) {
    const atIndex = email.indexOf("@");
    if (atIndex === -1) {
      return email; // Return the original email if "@" is not found
    }

    const modifiedPart = "*".repeat(atIndex);
    modifiedEmail = modifiedPart + email.slice(atIndex);
  }
  return modifiedEmail;
};
export function convertNumbersToAsterisks(text) {
  let return_vlaue = "";
  if (text) {
    return_vlaue = text.replace(/\d+/g, (match) => "*".repeat(match.length));
  }
  return return_vlaue;
}

export function formatDate(inputDate) {
  if (inputDate) {
    const date = new Date(inputDate);
    const day = date.getDate().toString().padStart(2, "0");
    const month = (date.getMonth() + 1).toString().padStart(2, "0"); // Month is 0-indexed
    const year = date.getFullYear();

    return `${day}/${month}/${year}`;
  }
}

export function createGoogleMapsURL(address) {
  // Encode the address for the URL
  const encodedAddress = encodeURIComponent(address);

  // Create the Google Maps URL
  const googleMapsURL = `https://www.google.com/maps/search/?api=1&query=${encodedAddress}`;

  return googleMapsURL;
}

export function ensureHttps(url) {

  return url; // URL already has "https://"
}

export function calculateDateDifference(startDate, endDate) {
  if (startDate && endDate) {
    // Define the two dates
    const date1 = new Date(startDate);
    const date2 = new Date(endDate);

    // Calculate the difference in milliseconds
    const differenceInMilliseconds = Math.abs(date2 - date1);

    // Convert the difference to days
    const differenceInDays = differenceInMilliseconds / (1000 * 60 * 60 * 24);
    if (differenceInDays == 0) {
      return 1;
    } else {
      return differenceInDays;
    }
  } else {
    return "-";
  }
}

export function isDateSmallerThanCurrent(dateString) {
  if (dateString) {
    const givenDate = new Date(dateString);

    // Check if givenDate is a valid date
    if (isNaN(givenDate.getTime())) {
      console.error("Invalid date string:", dateString);
      return false;
    }

    const currentDate = new Date();
    // console.log(currentDate, dateString);
    return givenDate < currentDate;
  }

  return false;
}

export function currentDateFormat() {
  //const checkEndDate = new Date().toISOString(); // Convert the Date object to a string
  const dateObject = new Date();
  // console.log(dateObject)
  // Format the Date object to a string in the desired format
  let formattedDate = dateObject.toLocaleDateString("en-GB");
  // console.log(formattedDate)
  return formattedDate;
}

export function removeHtmlTags(str) {
  if (typeof str === "string" || str instanceof String) {
    return str.replace(/<[^>]*>/g, "");
  } else {
    return "";
  }
}

export function formatText(text) {
  // Convert the entire string to lowercase
  const lowercasedText = text.toLowerCase();

  // Capitalize the first letter
  const formattedText =
    lowercasedText.charAt(0).toUpperCase() + lowercasedText.slice(1);

  return formattedText;
}
export const shouldShowMore = (content) => {
  const plainText = removeHtmlTags(content);
  return plainText?.length > 350; // Adjust character limit as needed
};

export const getRemainingContent = (content) => {
  const plainText = removeHtmlTags(content);
  return plainText?.substring(350); // Adjust to match where 6 lines end
};
export const splitContentIntoParagraphs = (content, charsPerLine = 100, minLines = 8) => {
  const plainText = removeHtmlTags(content);
  if (!plainText) return [];
  
  const minChars = charsPerLine * minLines;
  const paragraphs = [];
  let remainingText = plainText;
  
  while (remainingText.length > 0) {
    // Find optimal split point (end of sentence nearest to minChars)
    let splitPos = Math.min(minChars, remainingText.length);
    
    // Try to split at sentence boundaries
    const nextSentenceEnd = remainingText.substring(splitPos).match(/[.!?]\s/);
    if (nextSentenceEnd && nextSentenceEnd.index + splitPos < minChars * 1.5) {
      splitPos += nextSentenceEnd.index + 1;
    } else {
      // Fallback to word boundary
      const lastSpace = remainingText.lastIndexOf(' ', splitPos);
      if (lastSpace > minChars * 0.8) {
        splitPos = lastSpace;
      }
    }
    
    paragraphs.push(remainingText.substring(0, splitPos).trim());
    remainingText = remainingText.substring(splitPos).trim();
  }
  
  return paragraphs;
};
